<?php

namespace App\Console\Commands;

use App\Mail\SubscriptionExpirationMail;
use App\Models\Subscription;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class CheckSubscriptionExpiration extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'subscription:check-expiration';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check for expiring or expired subscriptions and send email notifications';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Checking subscription expirations...');

        // Get the notification days setting (default to 7 if not set)
        $notificationDays = getSuperAdminSettingValue('plan_expire_notification') ?? 7;

        // Get all active subscriptions
        $subscriptions = Subscription::with(['plan'])
            ->where('status', Subscription::ACTIVE)
            ->get();

        $expiringCount = 0;
        $expiredCount = 0;
        $errorCount = 0;

        foreach ($subscriptions as $subscription) {
            try {
                // Determine expiration date
                $expirationDate = $subscription->trial_ends_at
                    ? Carbon::parse($subscription->trial_ends_at)
                    : Carbon::parse($subscription->ends_at);

                $now = Carbon::now();
                $diffInDays = $now->diffInDays($expirationDate, false); // false = don't return absolute value

                // Check if subscription is expired
                $isExpired = $expirationDate->isPast();

                // Check if subscription is expiring soon (within notification days)
                $isExpiringSoon = !$isExpired && $diffInDays <= $notificationDays && $diffInDays > 0;

                if ($isExpired || $isExpiringSoon) {
                    // Get the primary user for this tenant (usually the admin/owner)
                    $user = User::whereTenantId($subscription->tenant_id)
                        ->where('is_active', User::IS_ACTIVE)
                        ->whereNotNull('email_verified_at')
                        ->first();

                    if ($user) {
                        // Send email notification
                        Mail::to($user->email)->send(
                            new SubscriptionExpirationMail(
                                $subscription,
                                $user,
                                $isExpired ? 0 : $diffInDays,
                                $isExpired
                            )
                        );

                        if ($isExpired) {
                            $expiredCount++;
                            $this->info("Sent expired notification to: {$user->email} (Subscription ID: {$subscription->id})");
                        } else {
                            $expiringCount++;
                            $this->info("Sent expiring notification to: {$user->email} (Subscription ID: {$subscription->id}, Days: {$diffInDays})");
                        }
                    } else {
                        $this->warn("No active user found for tenant_id: {$subscription->tenant_id}");
                    }
                }
            } catch (\Exception $e) {
                $errorCount++;
                Log::error('Error checking subscription expiration', [
                    'subscription_id' => $subscription->id,
                    'error' => $e->getMessage()
                ]);
                $this->error("Error processing subscription ID {$subscription->id}: {$e->getMessage()}");
            }
        }

        $this->info("Completed! Expiring: {$expiringCount}, Expired: {$expiredCount}, Errors: {$errorCount}");

        return Command::SUCCESS;
    }
}

