<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Plan;
use App\Models\User;
use App\Models\Setting;
use Laracasts\Flash\Flash;
use App\Models\Subscription;
use Illuminate\Http\Request;
use App\Models\AffiliateUser;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Contracts\View\View;
use App\Mail\ManualPaymentGuideMail;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Contracts\View\Factory;
use Stripe\Exception\ApiErrorException;
use Illuminate\Support\Facades\Redirect;
use App\Mail\SuperAdminManualPaymentMail;
use App\Repositories\SubscriptionRepository;
use Illuminate\Contracts\Foundation\Application;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException;
use App\Models\CardRequest;
use Illuminate\Support\Str;
use App\Models\Transaction;
use Illuminate\Support\Facades\Log;

class SubscriptionController extends AppBaseController
{
    private SubscriptionRepository $subscriptionRepo;

    public function __construct(SubscriptionRepository $subscriptionRepo)
    {
        $this->subscriptionRepo = $subscriptionRepo;
    }

    /**
     * @return Application|Factory|View
     */
    public function index(Request $request): \Illuminate\View\View
    {
        $currentPlan = getCurrentSubscription();

        $days = $remainingDay = '';
        if ($currentPlan->ends_at > Carbon::now()) {
            $days = Carbon::parse($currentPlan->ends_at)->diffInDays();
            $remainingDay = $days.' Days';
        }

        if ($days >= 30 && $days <= 365) {
            $remainingDay = '';
            $months = floor($days / 30);
            $extraDays = $days % 30;
            if ($extraDays > 0) {
                $remainingDay .= $months.' Month '.$extraDays.' Days';
            } else {
                $remainingDay .= $months.' Month ';
            }
        }

        if ($days >= 365) {
            $remainingDay = '';
            $years = floor($days / 365);
            $extraMonths = floor($days % 365 / 30);
            $extraDays = floor($days % 365 % 30);
            if ($extraMonths > 0 && $extraDays < 1) {
                $remainingDay .= $years.' Years '.$extraMonths.' Month ';
            } elseif ($extraDays > 0 && $extraMonths < 1) {
                $remainingDay .= $years.' Years '.$extraDays.' Days';
            } elseif ($years > 0 && $extraDays > 0 && $extraMonths > 0) {
                $remainingDay .= $years.' Years '.$extraMonths.' Month '.$extraDays.' Days';
            } else {
                $remainingDay .= $years.' Years ';
            }
        }

        return view('subscription.index', compact('currentPlan', 'remainingDay'));
    }

    public function choosePaymentType($planId, $context = null, $fromScreen = null)
    {
        $cashPaymentPlan = Subscription::where('tenant_id',Auth::user()->tenant_id)->where('payment_type','cash')->where('status',Subscription::PENDING)->first();

        if($cashPaymentPlan){
            Flash::error(__('messages.wait_for_apporove_of_cash_payment_by_admin'));

            return Redirect::back();
        }

        $appId     = config('services.onepay.appid');
        $hashToken = config('services.onepay.hash_token');
        $appToken  = config('services.onepay.app_token');
        $currency  = config('services.onepay.currency', 'LKR');
        $tranId = strtoupper(Str::random(10));
        $subscriptionsPricingPlan = Plan::findOrFail($planId);
        $price = $subscriptionsPricingPlan->price;
        $newPlan = getProratedPlanData($subscriptionsPricingPlan->id);
        $amount = getCurrencyAmount($newPlan['amountToPay'], $currency);
        $numericAmount = preg_replace('/[^\d.]/', '', $amount); // keeps only numbers & decimal

        $onePayData = [
			'appid' => $appId,
			'hashToken' => $hashToken,
			'amount' => (int) $numericAmount,
			'orderReference' => $tranId,
			'customerFirstName'      => Auth::user()->first_name,
            'customerLastName'       => Auth::user()->last_name,
			'customerPhoneNumber' => '+94771234567',
			'customerEmail' => Auth::user()->email,
			'transactionRedirectUrl'=> 'https://ebizcard.lemtoj.com/admin/dashboard',
			'additionalData' => 'returndata',
			'apptoken' => $appToken,
			'currency' => $currency,
		    ];

        // code for checking the current plan is active or not, if active then it should not allow to choose that plan
        $subscriptionsPricingPlan = Plan::findOrFail($planId);
        $paymentTypes = getPaymentGateway();


            return view('subscription.payment_for_plan', compact('subscriptionsPricingPlan', 'paymentTypes','onePayData'));
    }

    /**
     * @return Application|Factory|View
     */
    public function upgrade(){

        $user = Auth::user();

        $planActive = Plan::whereHas('subscriptions', function ($q) use ($user) {
                $q->where('tenant_id', $user->tenant_id);
            })
            ->first();

        if ($planActive && $planActive->plan_type == 1) {
            return redirect()->to('/admin/choose-payment-type/'. $planActive->id);
        }

        $cashPaymentPlan = Subscription::where('tenant_id',Auth::user()->tenant_id)->where('payment_type','cash')->where('status',Subscription::PENDING)->first();

        if($cashPaymentPlan && !$cashPaymentPlan->isExpired()){
            Flash::error(__('messages.wait_for_apporove_of_cash_payment_by_admin'));

            return Redirect::back();
        }
         // Get user's currency preference
        $userCurrencyPreference = Auth::user()->currency_preference ?? 'USD';
        // Handle backward compatibility: convert 'default' to 'USD'
        if ($userCurrencyPreference === 'default') {
            $userCurrencyPreference = 'USD';
        }

        // Filter plans based on currency preference
        $plansQuery = Plan::with(['currency', 'planFeature'])->whereStatus(Plan::IS_ACTIVE);

        if ($userCurrencyPreference === 'LKR') {
            // Show only LKR plans
            $plansQuery->whereHas('currency', function($q) {
                $q->where('currency_code', 'LKR');
            });
        } else {
            // Show all plans except LKR (default)
            $plansQuery->whereHas('currency', function($q) {
                $q->where('currency_code', '!=', 'LKR');
            });
        }

        $plans = $plansQuery->get();

        $monthlyPlans = $plans->where('frequency', Plan::MONTHLY);
        $yearlyPlans = $plans->where('frequency', Plan::YEARLY);
        $unLimitedPlans = $plans->where('frequency', Plan::UNLIMITED);

        return view(
            'subscription.upgrade',
            compact('monthlyPlans', 'yearlyPlans', 'unLimitedPlans')
        );
    }

    public function setPlanZero(Plan $plan): JsonResponse
    {
        try {
            DB::beginTransaction();

            Subscription::whereTenantId(getLogInTenantId())
                ->whereIsActive(true)->update(['is_active' => false]);

            $expiryDate = setExpiryDate($plan);
            Subscription::create([
                'plan_id' => $plan->id,
                'ends_at' => $expiryDate,
                'status' => true,
            ]);

            DB::commit();

            return $this->sendSuccess(__('messages.placeholder.subscribed_plan'));
        } catch (\Exception $e) {
            DB::rollBack();

            throw new UnprocessableEntityHttpException($e->getMessage());
        }
    }

    public function manualPay(Request $request)
    {
        $input = $request->all();
        if (isset($input['attachment']) && $input['attachment']->getClientMimeType() == 'text/php') {

            return $this->sendError('PHP file is not valid type for attachment');
        }
        $this->subscriptionRepo->manageSubscription($input);
        $data = Subscription::whereTenantId(getLogInTenantId())->orderBy('created_at', 'desc')->first();
        Subscription::whereId($data->id)->update(['payment_type' => 'Cash','status' => Subscription::PENDING]);
        $is_on = Setting::where('key', 'is_manual_payment_guide_on')->first();
        $manual_payment_guide_step = Setting::where('key', 'manual_payment_guide')->first();
        $user = \Illuminate\Support\Facades\Auth::user();
        $super_admin_data = [
            'super_admin_msg' => $user->full_name.' created request for payment of '.$data->plan->currency->currency_icon.''.$data->payable_amount,
            'attachment' => $data->attachment ?? '',
            'notes' => $data->notes ?? '',
            'id' => $data->id,
        ];

        if ($is_on['value'] == '1') {
            Mail::to($user['email'])
                ->send(new ManualPaymentGuideMail($manual_payment_guide_step['value'], $user));

            Mail::to('sadmin@vcard.com')
                ->send(new SuperAdminManualPaymentMail($super_admin_data, 'sadmin@vcard.com'));
        }

        return $this->sendSuccess(__('messages.placeholder.subscribed_plan_wait'));
    }

    public function downloadAttachment($id): \Illuminate\Http\Response|Application|\Illuminate\Contracts\Routing\ResponseFactory
    {
        $subscription = Subscription::findOrFail($id);

        [$file, $headers] = $this->subscriptionRepo->downloadAttachment($subscription);

        return response($file, 200, $headers);
    }

    public function downloadMailAttachment($id): BinaryFileResponse
    {
        $subscription = Subscription::whereTenantId(getLogInTenantId())->findOrFail($id);

        $headers = [
            'Content-Type' => $subscription->media[0]->mime_type,
            'Content-Description' => 'File Transfer',
            'Content-Disposition' => "attachment; filename={$subscription->media[0]->file_name}",
            'filename' => $subscription->media[0]->file_name,
        ];

        return response()->download($subscription->media[0]->getPath(), $subscription->media[0]->file_name, $headers);
    }

    /**
     * @return Application|Factory|View
     */
    public function cashPlan(): \Illuminate\View\View
    {
        return view('sadmin.planPyment.index');
    }

    public function planStatus(Request $request): JsonResponse
    {
        if ($request->status == Subscription::ACTIVE) {
            Subscription::whereTenantId($request->tenant_id)
                ->where('id', '!=', $request->id)
                ->where('status', '!=', Subscription::REJECT)
                ->update(['status' => Subscription::INACTIVE]);
        }

        Subscription::where('id', $request->id)->update([
            'status' => $request->status,
        ]);
        $subscription = Subscription::where('id', $request->id)->first();

        $user = User::whereTenantId($request->tenant_id)->first();

        AffiliateUser::whereUserId($user->id)
            ->withoutGlobalScopes()
            ->update(['is_verified' => 1]);

        if ($request->status == Subscription::ACTIVE) {
            // generate transaction id
            $txnId = uniqid(''); // or Str::uuid()
            $payableAmount = $request->payable_amount ?? 0;
            // create transaction
            Transaction::create([
                'transaction_id' => $txnId,
                'amount'         => $subscription->payable_amount ?? 0,
                'type'           => 1,
                'tenant_id'      => $request->tenant_id,
                'status'         => Transaction::SUCCESS,
                'meta'           => json_encode([
                    'payment_status_code' => 0,
                    'total_amount'        => $payableAmount,
                    'subscription_id'     => $request->id,
                ]),
            ]);

            // update subscription with same transaction id
            Subscription::where('id', $request->id)->update([
                'transaction_id' => $txnId,
            ]);
        }

        return $this->sendSuccess(__('messages.placeholder.payment_received'));
    }

    public function purchaseSubscription(Request $request)
    {
        $input = $request->all();
        $result = $this->subscriptionRepo->purchaseSubscriptionForStripe($input);
        // Always update vcard aliases using tenant_id
        if (!empty($result['user_id']) && !empty($result['tenant_id'])) {
            $this->updateNfcOrderedVcardUrlAlias($result['tenant_id'], $result['user_id']);

        }

        if (isset($result['status']) && $result['status'] === true) {
            return $this->sendSuccess(
                $result['subscriptionPlan']->name . ' ' . __('messages.subscription.has_been_subscribed')
            );
        }

        if (isset($result['status']) && $result['status'] === false) {
            return $this->sendError(__('messages.placeholder.cannot_switch_to_zero'));
        }

        return $this->sendResponse($result, 'Session created successfully.');
    }

    private function updateNfcOrderedVcardUrlAlias(string $tenantId, int $userId): int
    {
        // Get vcard IDs from NFC orders for this user
        $orderedVcardIds = DB::table('nfc_orders')
            ->where('user_id', $userId)
            ->pluck('vcard_id')
            ->toArray();

        if (empty($orderedVcardIds)) {
            Log::info("updateNfcOrderedVcardUrlAlias: no NFC ordered vcards for user {$userId}, tenant {$tenantId}");
            return 0;
        }

        // Fetch only those vcards
        $vcards = DB::table('vcards')
            ->where('tenant_id', $tenantId)
            ->whereIn('id', $orderedVcardIds)
            ->get();

        if ($vcards->isEmpty()) {
            Log::info("updateNfcOrderedVcardUrlAlias: no vcards found for ordered IDs in tenant {$tenantId}");
            return 0;
        }

        $updated = 0;

        DB::transaction(function () use ($vcards, &$updated) {
            foreach ($vcards as $vcard) {
                $currentAlias = trim((string) ($vcard->url_alias ?? ''));

                // Build base alias from first_name + last_name (fallback to vcard-{id})
                $first = trim((string) ($vcard->first_name ?? ''));
                $last  = trim((string) ($vcard->last_name ?? ''));
                $name  = trim($first . ' ' . $last);

                if ($name === '') {
                    $name = 'vcard-' . $vcard->id;
                }

                $baseAlias = Str::slug($name, '-');
                $alias = $baseAlias;
                $counter = 1;

                // Ensure uniqueness across url_alias and old_url_alias excluding current record
                while (
                    DB::table('vcards')
                        ->where(function ($q) use ($alias) {
                            $q->where('url_alias', $alias)
                            ->orWhere('old_url_alias', $alias);
                        })
                        ->where('id', '!=', $vcard->id)
                        ->exists()
                ) {
                    $alias = $baseAlias . '-' . $counter++;
                }

                // Prepare update payload
               // Prepare update payload
                $payload = [];

                // If current alias exists, move it to old_url_alias
                if (empty($vcard->old_url_alias) && $currentAlias !== '') {
                    $payload['old_url_alias'] = $currentAlias;

                    // If new alias ended up same as old, regenerate it
                    if ($alias === $currentAlias) {
                        $alias = $baseAlias . '-' . $counter++;
                        while (
                            DB::table('vcards')
                                ->where(function ($q) use ($alias) {
                                    $q->where('url_alias', $alias)
                                    ->orWhere('old_url_alias', $alias);
                                })
                                ->where('id', '!=', $vcard->id)
                                ->exists()
                        ) {
                            $alias = $baseAlias . '-' . $counter++;
                        }
                    }
                }

                // Always update url_alias
                $payload['url_alias'] = $alias;

                DB::table('vcards')->where('id', $vcard->id)->update($payload);
                $updated++;
            }
        });

        Log::info("updateNfcOrderedVcardUrlAlias: updated {$updated} vcards for user {$userId}, tenant {$tenantId}");

        return $updated;
    }

    /**
     * @return Application|Factory|View
     *
     * @throws ApiErrorException
     */
    public function paymentSuccess(Request $request): \Illuminate\View\View
    {
        AffiliateUser::whereUserId(getLogInUserId())->withoutGlobalScopes()
            ->update(['is_verified' => 1]);

        /** @var SubscriptionRepository $subscriptionRepo */
        $subscriptionRepo = app(SubscriptionRepository::class);
        $subscription = $subscriptionRepo->paymentUpdate($request);
        Flash::success($subscription->plan->name.' '.__('messages.subscription.has_been_subscribed'));

        return view('sadmin.plans.payment.paymentSuccess');
    }

    public function handleFailedPayment(): \Illuminate\View\View
    {
        $subscriptionPlanId = session('subscription_plan_id');
        /** @var SubscriptionRepository $subscriptionRepo */
        $subscriptionRepo = app(SubscriptionRepository::class);
        $subscriptionRepo->paymentFailed($subscriptionPlanId);
        Flash::error(__('messages.placeholder.unable_to_process_payment'));

        return view('sadmin.plans.payment.paymentcancel');
    }

    /**
     * @return Application|Factory|View
     */
    public function userSubscribedPlan(): \Illuminate\View\View
    {
        return view('sadmin.subscriptionPlan.index');
    }

    public function userSubscribedPlanEdit(Request $request): JsonResponse
    {
        $subscription = Subscription::with([
            'plan:id,name,currency_id', 'tenant.user', 'plan.currency',
        ])->whereId($request->id)->first();

        return $this->sendResponse($subscription, 'Subscription successfully retrieved.');
    }

    public function userSubscribedPlanUpdate(Request $request): JsonResponse
    {
        if(empty($request->end_date)){
            return $this->sendError(__('messages.subscription.end_date_required'));
        }
        $subscription = Subscription::where('id', $request->id)->update([
            'ends_at' => $request->end_date,
            'status' => Subscription::ACTIVE,
        ]);

        return $this->sendResponse($subscription, __('messages.placeholder.subscription_date_updated'));
    }

     public function checkStatus(Request $request)
    {
        $subscriptionId = $request->query('subscription_id');
        $subscription = Subscription::find($subscriptionId);

        if (!$subscription) {
            return response()->json(['status' => 'not_found']);
        }

        return response()->json(['status' => $subscription->status]);
    }

    /**
     * Check subscription expiry status using isSubscriptionExpired() helper
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function checkSubscriptionExpiry(Request $request): JsonResponse
    {
        try {
            $result = isSubscriptionExpired();

            return response()->json([
                'success' => true,
                'data' => $result,
                'timestamp' => now()->toDateTimeString(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'timestamp' => now()->toDateTimeString(),
            ], 500);
        }
    }
     public function calculatePlanPrice($requestedCards)
    {
        $user = auth()->user();

        $subscription = $user->subscription;
        $plan = $subscription->plan;

        $perCardYearly = $plan->per_card_price; // e.g., $20

        // Daily price
        $pricePerCardDaily = $perCardYearly / 365;

        // Subscription end date
        $today     = now();

        $startDate = Carbon::parse($subscription->starts_at);
        $endDate   = Carbon::parse($subscription->ends_at);

        $totalDays = $startDate->diffInDays($endDate) + 1;
        $remainingDays = $today->diffInDays($endDate) + 1;

        // Total amount = daily price × days left × cards
        $amount = round($pricePerCardDaily * $remainingDays * $requestedCards, 2);

        return ['amount' => $amount, 'perCardYearly' => $perCardYearly, 'startDate' => $startDate, 'endDate' => $endDate, 'pricePerCardDaily' => $pricePerCardDaily, 'remainingDays' => $remainingDays];
    }
     public function calculateAmount(Request $request)
    {

        $requestedCards = (int) $request->requested_cards;
        $price_data = $this->calculatePlanPrice($requestedCards);

        $user = auth()->user();
        $subscription = $user->subscription;

        return response()->json([
            'subscription' => $subscription->id,
            'endDate' => $price_data['endDate'],
            'perCardYearly' => $price_data['perCardYearly'],
            'pricePerCardDaily' => $price_data['pricePerCardDaily'],
            'requested_cards' => $requestedCards,
            'remaining_days' => $price_data['remainingDays'],
            'amount' => round($price_data['amount'], 2)
        ]);
    }
     public function requestCardStore(Request $request): JsonResponse
    {
        try {
            DB::beginTransaction();

            $request->validate([
                'requested_cards' => 'required|integer|min:1',
            ]);

            $user = auth()->user();

            // Get subscription
            $subscription = $user->subscription;

            if (!$subscription) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active subscription found!',
                ], 404);
            }

            $plan = $subscription->plan;

            if (!$plan) {
                return response()->json([
                    'success' => false,
                    'message' => 'Plan not found!',
                ], 404);
            }

            if (!$subscription->starts_at || !$subscription->ends_at) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid subscription period!',
                ], 400);
            }

            $startDate = Carbon::parse($subscription->starts_at);
            $endDate   = Carbon::parse($subscription->ends_at);
            $today     = now();

            if ($today->gte($endDate)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Subscription already expired!',
                ], 400);
            }

            $requestedCards = (int) $request->requested_cards;
            $price_data = $this->calculatePlanPrice($requestedCards);

            $cardrequests = CardRequest::create([
                'tenant_id'       => $user->tenant_id,
                'user_id'         => $user->id,
                'subscription_id' => $plan->no_of_vcards + $requestedCards,
                'requested_cards' => $requestedCards,
                'amount'          => $price_data['amount'],
                'status'          => 0

            ]);
            /*
            Subscription::create([
                'plan_id' => $plan->id,
                'plan_amount' => $plan->price,
                'payable_amount' => $amount,
                'plan_frequency' => Plan::MONTHLY,
                'starts_at' => Carbon::now(),
                'ends_at' => Carbon::now()->addDays($plan->trial_days),
                'trial_ends_at' => Carbon::now()->addDays($plan->trial_days),
                'status' => Subscription::ACTIVE,
                'tenant_id' => $user->tenant_id,
                'no_of_vcards' => ($plan->no_of_vcards+$requestedCards),
            ]);
            */

            DB::commit();

            //return $this->sendSuccess(__('messages.placeholder.subscribed_plan'));

            return response()->json([
                'success'  => true,
                'message'  => __('messages.placeholder.subscribed_plan'),
                'payment_url' => route('choose.payment.type', $plan->id) . '?cardrequests=' . $cardrequests->id,
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();

            throw new UnprocessableEntityHttpException($e->getMessage());
        }
    }

public function requestDowngradeStore(Request $request): JsonResponse
{
    try {
        DB::beginTransaction();

        // Validate the requested number of cards for downgrade
        $request->validate([
            'requested_cards' => 'required|integer|min:1',
        ]);

        $user = auth()->user();

        // Get the active subscription
        $subscription = $user->subscription;

        if (!$subscription) {
            return response()->json([
                'success' => false,
                'message' => 'No active subscription found!',
            ], 404);
        }

          // Check if today is within the last 10 days before expiry
        $expiryDate = \Carbon\Carbon::parse($subscription->ends_at); // assuming you store expiry in ends_at
        $today = \Carbon\Carbon::now();
        $daysRemaining = $today->diffInDays($expiryDate, false);
        $allowedDays = config('subscription.downgrade_allowed_days');
        // if ($daysRemaining > $allowedDays) {
        //     return response()->json([
        //         'success' => false,
        //         'message' => 'You can only downgrade within the last 10 days of your current plan.',
        //     ], 400);
        // }

        $plan = $subscription->plan;

        if (!$plan) {
            return response()->json([
                'success' => false,
                'message' => 'Plan not found!',
            ], 404);
        }

        if (!$subscription->starts_at || !$subscription->ends_at) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid subscription period!',
            ], 400);
        }

        $startDate = Carbon::parse($subscription->starts_at);
        $endDate   = Carbon::parse($subscription->ends_at);
        $today     = now();

        if ($today->gte($endDate)) {
            return response()->json([
                'success' => false,
                'message' => 'Subscription already expired!',
            ], 400);
        }

        $requestedCards = (int) $request->requested_cards;
        $price_data = $this->calculateDowngradePrice($requestedCards);

        // Ensure we don't downgrade below 0 cards
        $newCardCount = $plan->no_of_vcards - $requestedCards;
        if ($newCardCount < 0) {
            return response()->json([
                'success' => false,
                'message' => 'Requested downgrade exceeds the current card count!',
            ], 400);
        }

        // Check if we need to disable vCards before downgrade
        $vcardsToDisable = $requestedCards;

        // Fetch active vCards with status 1 (enabled)
        // $activeVCards = $user->vcards()->where('status', 1)->take($vcardsToDisable)->get();

        $createdVCards = $user->vcards()->count();

        // Count how many are active (status = 1)
        $activeVCards = $user->vcards()->where('status', 1)->count();
        // dd($activeVCards);
        // Case 1: User has created all vCards → they must disable before downgrade
            if ($activeVCards > $newCardCount) {
               return response()->json([
                    'success' => false,
                    'message' => "Please disable {$requestedCards} vCards before downgrading!",
                ], 400);
            }

        // Update the subscription to reflect the downgraded card count
        $plan->update([
            'no_of_vcards' => $newCardCount,
            'price' => $price_data['remaing_amount'],
        ]);

        // Create a new card request entry to track the downgrade
        $cardrequests = CardRequest::create([
            'tenant_id'       => $user->tenant_id,
            'user_id'         => $user->id,
            'subscription_id' => $subscription->id, // Use actual subscription ID
            'requested_cards' => $requestedCards,
            'amount'          => $price_data['amount'],
            'status'          => 0 // Pending status
        ]);

        DB::commit();

        return response()->json([
            'success'  => true,
            'message'  => __('messages.placeholder.downgraded_plan'),
            'payment_url' => route('choose.payment.type', $plan->id) . '?cardrequests=' . $cardrequests->id,
        ], 200);
    } catch (\Exception $e) {
        DB::rollBack();

        return response()->json([
            'success' => false,
            'message' => 'An error occurred while processing the downgrade request.',
            'error' => $e->getMessage(),
        ], 500);
    }
}

public function calculateDowngradeAmount(Request $request)
{
    try {
        // 1. Get the requested number of cards for downgrade
        $requestedCards = (int) $request->requested_cards;

        // 2. Calculate the price based on the requested downgrade
        $price_data = $this->calculateDowngradePrice($requestedCards);

        // 3. Get the currently authenticated user
        $user = auth()->user();

        // 4. Retrieve the user's active subscription
        $subscription = $user->subscription;

        if (!$subscription) {
            return response()->json([
                'success' => false,
                'message' => 'No active subscription found!',
            ], 404);
        }

        // 5. Ensure that the subscription is valid
        $startDate = Carbon::parse($subscription->starts_at);
        $endDate = Carbon::parse($subscription->ends_at);
        $today = now();

        if ($today->gte($endDate)) {
            return response()->json([
                'success' => false,
                'message' => 'Subscription has expired!',
            ], 400);
        }

        // 6. Calculate remaining days
        $remainingDays = $today->diffInDays($endDate);

        // 7. Adjust the number of cards (downgrade logic)
        $newCardCount = $subscription->no_of_vcards - $requestedCards;
        if ($newCardCount < 0) {
            return response()->json([
                'success' => false,
                'message' => 'Requested downgrade exceeds the current card count!',
            ], 400);
        }

        // 8. Prepare the response data
        return response()->json([
            'success' => true,
            'subscription_id' => $subscription->id,
            'remaining_days' => $remainingDays,
            'requested_cards' => $requestedCards,
            'new_card_count' => $newCardCount,
            'amount' => round($price_data['amount'], 2),
            'per_card_price' => $price_data['perCardPrice'], // Price per card
        ]);
    } catch (\Exception $e) {
        // Handle any exceptions
        return response()->json([
            'success' => false,
            'message' => 'An error occurred: ' . $e->getMessage(),
        ], 500);
    }
}

public function calculateDowngradePrice(int $requestedCards)
{
    // Get the current user's active subscription plan
    $user = auth()->user();
    $subscription = $user->subscription;
    $plan = $subscription->plan;

    // Check if the subscription exists
    if (!$subscription || !$subscription->plan) {
        throw new \Exception('No valid plan associated with the user!');
    }

    // Retrieve the price per card from the Plan model
    $pricePerCard = $plan->per_card_price;// Assuming 'price_per_card' exists in your Plan model

    // Calculate the amount to be reduced by removing the requested number of cards
    $currentCardCount = $subscription->no_of_vcards; // Current number of cards in the subscription
    $cardsToRemove = $requestedCards;

    // Ensure we don't remove more cards than the user has
    if ($cardsToRemove > $currentCardCount) {
        throw new \Exception('Requested downgrade exceeds the current card count!');
    }

    // Calculate the total price reduction for the cards being removed
    $priceReduction = $pricePerCard * $cardsToRemove;

    $price=$subscription->plan_amount;
    // Calculate the new price after removing the cards
    $newAmount = $subscription->plan_amount - $priceReduction;
    $plan = $subscription->plan;
    // Update the subscription's amount with the new reduced price
    // $plan->update([
    //     'amount' => $newAmount, // Update the price with the new reduced amount
    // ]);

    // Return the calculated price data
    return [
        'price'=>$price,
        'amount' => 0, // New amount after removing the cards
        'remaing_amount'=>$newAmount,
        'perCardPrice' => $pricePerCard, // The price per card
        'priceReduction' => $priceReduction, // How much the price was reduced by
    ];
}


}
