<?php

namespace App\Http\Controllers;

use Str;
use URL;
use Exception;
use Carbon\Carbon;
use App\Models\Vcard;
use Spatie\Color\Hex;
use App\Models\Product;
use App\Models\Setting;
use App\Models\Currency;
use App\Models\VcardBlog;
use App\Models\QrcodeEdit;
use App\Models\SocialIcon;
use App\Models\SocialLink;
use Laracasts\Flash\Flash;
use App\Models\Appointment;
use App\Models\UserSetting;
use App\Models\Subscription;
use Illuminate\Http\Request;
use App\Models\PrivacyPolicy;
use App\Models\TermCondition;
use App\Models\VcardSections;
use App\Models\AppointmentDetail;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Redirector;
use App\Models\ScheduleAppointment;
use Illuminate\Contracts\View\View;
use App\Exports\VcardTemplateExport;
use Maatwebsite\Excel\Facades\Excel;
use App\Repositories\VcardRepository;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Contracts\View\Factory;
use Illuminate\Support\Facades\Session;
use App\Http\Requests\CreateVcardRequest;
use App\Http\Requests\UpdateVcardRequest;
use JeroenDesloovere\VCard\VCard as VCardVCard;
use Illuminate\Contracts\Foundation\Application;
use Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException;
use Illuminate\Support\Facades\Auth;
use App\Imports\VcardsImport;
use Firebase\JWT\JWT;
use Firebase\JWT\Key;
use Firebase\JWT\ExpiredException;
use App\Models\UserPrivacySetting;
use App\Models\CardAccessToken;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class VcardController extends AppBaseController
{
    private VcardRepository $vcardRepository;

    public function __construct(VcardRepository $vcardRepository)
    {
        $this->vcardRepository = $vcardRepository;
    }

    /**
     * @return Application|Factory|View
     */
    public function index(): \Illuminate\View\View
    {
        $makeVcard = $this->vcardRepository->checkTotalVcard();
         $currentSubscription = getCurrentSubscription();
        return view('vcards.index', compact('makeVcard','currentSubscription'));
    }

    /**
     * @return Application|Factory|View
     */
    public function template(): \Illuminate\View\View
    {
        return view('sadmin.vcards.index');
    }

    public function download($id): JsonResponse
    {
        $data = Vcard::with('socialLink')->find($id);

        return $this->sendResponse($data, __('messages.flash.vcard_retrieve'));
    }

    /**
     * @return Application|Factory|View
     */
    public function vcards(): \Illuminate\View\View
    {
        $makeVcard = $this->vcardRepository->checkTotalVcard();

        return view('vcards.templates', compact('makeVcard'));
    }

    /**
     * @return Application|Factory|View
     */
    public function create()
    {
        $makeVcard = $this->vcardRepository->checkTotalVcard();
        if (! $makeVcard) {
            return redirect(route('vcards.index'));
        }

        $partName = 'basics';

        return view('vcards.create', compact('partName'));
    }

    /**
     * @return Application|RedirectResponse|Redirector
     */
    public function store(CreateVcardRequest $request): RedirectResponse
    {
        $input = $request->all();

        $vcard = $this->vcardRepository->store($input);

        Flash::success(__('messages.flash.vcard_create'));

        return redirect(route('vcards.edit', $vcard->id));
    }

    /**
     * @return Application|Factory|View
     */




    public function show($alias, $id = null): \Illuminate\View\View
    {

        $vcard = Vcard::with([
            'businessHours' => function ($query) {
                $query->where('end_time', '!=', '00:00');
            },
            'services',
            'testimonials',
            'products',
            'blogs',
            'privacy_policy',
            'term_condition',
            'user',
            'template'
        ])->whereUrlAlias($alias)->first();
        // dd($vcard->term_condition);
        $vcardProducts = $vcard->products->sortDesc()->take(6);

        $blogSingle = '';
        if (isset($id)) {
            $blogSingle = VcardBlog::where('id', $id)->first();
        }

        $setting = Setting::pluck('value', 'key')->toArray();

        // ✅ Build $reqpage first
        $reqpage = str_replace('/' . $vcard->url_alias, '', \Request::getRequestUri());
        $reqpage = empty($reqpage) ? 'index' : $reqpage;
        $reqpage = preg_replace("/\.$/", '', $reqpage);
        $reqpage = preg_replace('/[0-9]+/', '', $reqpage);
        $reqpage = str_replace('/', '', $reqpage);
        $reqpage = str_contains($reqpage, '?') ? substr($reqpage, 0, strpos($reqpage, '?')) : $reqpage;

        // ✅ Template name with safe fallback
        $vcard_name = $vcard->template?->name ?? 'vcard3';
        $vcard_name = $vcard_name === 'vcard11' ? 'vcard11.' . $reqpage : $vcard_name;

        $url = explode('/', $vcard->location_url);

        $appointmentDetail = AppointmentDetail::where('vcard_id', $vcard->id)->first();
        $managesection = VcardSections::where('vcard_id', $vcard->id)->first();

        $userSetting = UserSetting::where('user_id', $vcard->user->id)->pluck('value', 'key')->toArray();

        $currency = '';
        $paymentMethod = null;
        if (count($userSetting) > 0) {
            $currency = Currency::where('id', $userSetting['currency_id'])->first();
            $paymentMethod = getPaymentMethod($userSetting);
        }

        $businessDaysTime = [];
        if ($vcard->businessHours->count()) {
            $dayKeys = [1, 2, 3, 4, 5, 6, 7];
            $openDayKeys = [];
            $openDays = [];
            $closeDays = [];

            foreach ($vcard->businessHours as $key => $openDay) {
                $openDayKeys[] = $openDay->day_of_week;
                $openDays[$openDay->day_of_week] = $openDay->start_time . ' - ' . $openDay->end_time;
            }

            $closedDayKeys = array_diff($dayKeys, $openDayKeys);

            foreach ($closedDayKeys as $closeDayKey) {
                $closeDays[$closeDayKey] = null;
            }

            $businessDaysTime = $openDays + $closeDays;
            ksort($businessDaysTime);
        }

        $customQrCode = QrcodeEdit::whereTenantId($vcard->user->tenant_id)->pluck('value', 'key')->toArray();

        // Set default values if customQrCode is null or empty
        if ($customQrCode == null || empty($customQrCode)) {
            $customQrCode['qrcode_color'] = '#000000';
            $customQrCode['background_color'] = '#ffffff';
            $customQrCode['qrcode_opacity'] = '1';
            $customQrCode['background_opacity'] = '1';
        }

        // Ensure opacity values exist
        if (!isset($customQrCode['qrcode_opacity'])) {
            $customQrCode['qrcode_opacity'] = '1';
        }
        if (!isset($customQrCode['background_opacity'])) {
            $customQrCode['background_opacity'] = '1';
        }

        // Keep the original RGB conversion for backward compatibility
        $qrcodeColor['qrcodeColor'] = Hex::fromString($customQrCode['qrcode_color'])->toRgb();
        $qrcodeColor['background_color'] = Hex::fromString($customQrCode['background_color'])->toRgb();

        // Add RGBA support with opacity
        $qrcodeColor['qrcodeColorRgba'] = $this->getRgbaFromHex(
            $customQrCode['qrcode_color'],
            $customQrCode['qrcode_opacity']
        );
        $qrcodeColor['backgroundColorRgba'] = $this->getRgbaFromHex(
            $customQrCode['background_color'],
            $customQrCode['background_opacity']
        );

        // Add opacity values for easy access
        $qrcodeColor['qrcodeOpacity'] = floatval($customQrCode['qrcode_opacity']);
        $qrcodeColor['backgroundOpacity'] = floatval($customQrCode['background_opacity']);

        if (empty(getLocalLanguage())) {
            $alias = $vcard->url_alias;
            $languageName = $vcard->default_language;
            session(['languageChange_' . $alias => $languageName]);
            setLocalLang(getLocalLanguage());
        }
        // Determine if profile is locked
        $locked = $vcard->is_locked;

        $shareJwt = request('share');
        $tid = request('tid');
        $token = request('t');

        if(!empty($shareJwt)) {
            $result = $this->checkJwt($shareJwt);

            if ($result['status']) {
            $locked = false;
            } else {
            $locked = $locked;
            }
        }

        $tokens = CardAccessToken::where('vcard_id', $vcard->id)
                            ->where('revoked', false)
                            ->get();

        foreach ($tokens as $cat) {
            if (Hash::check($token, $cat->token_hash)) {
                $locked = false; // valid token found
                break;
            }
        }

        if ($locked) {
            // You may want to strip/hide sensitive relations
            $vcard->loadMissing('services');
            $vcard->makeHidden(['blogs', 'products', 'testimonials']);
        }
        if ($vcard->status) {
            $vcard_name = $vcard->template?->name ?? 'vcard3';
            return view(
                'vcardTemplates.' . $vcard_name,
                compact(
                    'vcard',
                    'setting',
                    'url',
                    'appointmentDetail',
                    'managesection',
                    'userSetting',
                    'currency',
                    'paymentMethod',
                    'blogSingle',
                    'businessDaysTime',
                    'customQrCode',
                    'qrcodeColor',
                    'vcardProducts',
                    'locked'
                )
            );
        }
        abort('404');
    }
    public function checkJwt($jwt)
    {
        try {
            // Try decoding the token
            $decoded = JWT::decode($jwt, new Key(env('JWT_SECRET'), 'HS256'));

            // ✅ If no exception → JWT is valid and not expired
            return [
                'status'  => true,
                'message' => 'JWT is valid',
                'data'    => (array) $decoded
            ];
        } catch (ExpiredException $e) {
            // ⛔ Token expired
            return [
                'status'  => false,
                'message' => 'JWT has expired'
            ];
        } catch (\Exception $e) {
            // ⛔ Invalid for other reasons (wrong signature, tampered, etc.)
            return [
                'status'  => false,
                'message' => 'JWT is invalid: ' . $e->getMessage()
            ];
        }
    }
    /**
     * Get RGBA array from hex color and opacity
     *
     * @param string $hex
     * @param float $opacity
     * @return array
     */
    private function getRgbaFromHex($hex, $opacity = 1)
    {
        // Remove # if present
        $hex = ltrim($hex, '#');

        // Convert hex to RGB
        if (strlen($hex) == 6) {
            return [
                'r' => hexdec(substr($hex, 0, 2)),
                'g' => hexdec(substr($hex, 2, 2)),
                'b' => hexdec(substr($hex, 4, 2)),
                'a' => floatval($opacity)
            ];
        }

        // Default to black with opacity if invalid hex
        return ['r' => 0, 'g' => 0, 'b' => 0, 'a' => floatval($opacity)];
    }

    /**
     * Get RGBA CSS string from RGBA array
     *
     * @param array $rgba
     * @return string
     */
    public function getRgbaCssString($rgba)
    {
        return "rgba({$rgba['r']}, {$rgba['g']}, {$rgba['b']}, {$rgba['a']})";
    }

    /**
     * Get RGB values with opacity applied for QR code libraries
     *
     * @param object $rgbObject - The RGB object from Hex::fromString()->toRgb()
     * @param float $opacity
     * @return array
     */
    public function applyOpacityToRgb($rgbObject, $opacity = 1)
    {
        return [
            'red' => $rgbObject->red(),
            'green' => $rgbObject->green(),
            'blue' => $rgbObject->blue(),
            'opacity' => floatval($opacity)
        ];
    }

    public function checkPassword(Request $request, Vcard $vcard): JsonResponse
    {
        setLocalLang(checkLanguageSession($vcard->url_alias));

        if (Crypt::decrypt($vcard->password) == $request->password) {
            session(['password_' => '1']);

            return $this->sendSuccess(__('messages.placeholder.password_is_correct'));
        }

        return $this->sendError(__('messages.placeholder.password_invalid'));
    }

    /**
     * @return Application|Factory|View|RedirectResponse|Redirector
     */
    public function edit(Vcard $vcard, Request $request)
    {
        $partName = ($request->part === null) ? 'basics' : $request->part;
        // dd($partName);
        if ($partName !== TermCondition::TERM_CONDITION && $partName !== PrivacyPolicy::PRIVACY_POLICY) {
            if (! checkFeature($partName)) {
                return redirect(route('vcards.edit', $vcard->id));
            }
        }

        $data = $this->vcardRepository->edit($vcard);
        $data['partName'] = $partName;

        $appointmentDetail = AppointmentDetail::where('vcard_id', $vcard->id)->first();
        $privacyPolicy    = PrivacyPolicy::where('vcard_id', $vcard->id)->first();
        $termCondition    = TermCondition::where('vcard_id', $vcard->id)->first();
        $managesection    = VcardSections::where('vcard_id', $vcard->id)->first();

        $user = Auth::user();

        // Fetch all privacy records for this user (optionally per vcard if you have vcard_id column)
        $privacyQuery = UserPrivacySetting::where('user_id', $user->id);

        // If you store privacy per vcard and your table has vcard_id, uncomment:
        // $privacyQuery->where('vcard_id', $vcard->id);

        $privacyRecords = $privacyQuery->latest()->get();

        // Provide the latest record as $privacy for form defaults
        $privacy = $privacyRecords->first();

        // safe default when no record exists (so optional($privacy)->... works)
        if (! $privacy) {
            $privacy = (object) [
                'is_locked'   => false,
                'expiry_type' => null,
                'custom_days' => null,
                'expires_at'  => null,
            ];
        }
         // Fetch Card Access Tokens for this vcard
        $cardAccessTokens = CardAccessToken::where('vcard_id', $vcard->id)
            ->latest()
            ->get();

        // ✅ Build QR code settings (same as in show())
        $customQrCode = QrcodeEdit::whereTenantId($vcard->user->tenant_id)->pluck('value', 'key')->toArray();
        if ($customQrCode == null || empty($customQrCode)) {
            $customQrCode['qrcode_color'] = '#000000';
            $customQrCode['background_color'] = '#ffffff';
            $customQrCode['qrcode_opacity'] = '1';
            $customQrCode['background_opacity'] = '1';
        }
        $customQrCode['qrcode_opacity'] = $customQrCode['qrcode_opacity'] ?? '1';
        $customQrCode['background_opacity'] = $customQrCode['background_opacity'] ?? '1';

        $qrcodeColor['qrcodeColor'] = Hex::fromString($customQrCode['qrcode_color'])->toRgb();
        $qrcodeColor['background_color'] = Hex::fromString($customQrCode['background_color'])->toRgb();
        $qrcodeColor['qrcodeColorRgba'] = $this->getRgbaFromHex(
            $customQrCode['qrcode_color'],
            $customQrCode['qrcode_opacity']
        );
        $qrcodeColor['backgroundColorRgba'] = $this->getRgbaFromHex(
            $customQrCode['background_color'],
            $customQrCode['background_opacity']
        );
        $qrcodeColor['qrcodeOpacity'] = floatval($customQrCode['qrcode_opacity']);
        $qrcodeColor['backgroundOpacity'] = floatval($customQrCode['background_opacity']);
        return view('vcards.edit', compact(
            'appointmentDetail',
            'privacyPolicy',
            'termCondition',
            'managesection',
            'privacy',
            'privacyRecords',
            'cardAccessTokens', // ✅ added
            'customQrCode',
            'qrcodeColor'
        ))->with($data);
    }

    public function generateCardAccessToken(Vcard $vcard)
    {
        $catData = $this->vcardRepository->generateCardAccessToken($vcard);

        return response()->json([
            'status'  => true,
            'message' => 'Card Access Token generated successfully',
            'data'    => $catData
        ]);
    }

    public function revokeCardToken(Request $request, $vcardId)
    {
        $tokenId = $request->token_id;
        $vcard = Vcard::findOrFail($vcardId);

        // if ($vcard->user_id !== auth()->id()) {
        //     return response()->json([
        //         'status' => false,
        //         'message' => 'Unauthorized action.',
        //     ], 403);
        // }

        $token = CardAccessToken::where('vcard_id', $vcardId)
            ->where('id', $tokenId)
            ->first();

        if (!$token) {
            return response()->json([
                'status' => false,
                'message' => 'Token not found.',
            ], 404);
        }

        if ($token->revoked) {
            return response()->json([
                'status' => false,
                'message' => 'This token is already revoked.',
            ]);
        }

        $token->update([
            'revoked' => true,
            'revoked_at' => now(),
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Card Access Token revoked successfully.',
        ]);
    }


    public function updateStatus(Vcard $vcard): JsonResponse
    {
        $tenantId = $vcard->tenant_id;

        // Get subscription for this tenant
        $subscription = \App\Models\Subscription::where('tenant_id', $tenantId)
            ->where('status', \App\Models\Subscription::ACTIVE)
            ->first();

        if (! $subscription) {
            return $this->sendError('No active subscription found.');
        }

        // Count currently active vcards
        $activeCount = Vcard::where('tenant_id', $tenantId)
            ->where('status', 1)
            ->count();

        $allowed = $subscription->no_of_vcards;

        // If we are trying to activate this one but already at limit → block it
        if ($vcard->status == 0 && $activeCount >= $allowed) {
            return $this->sendError(__('messages.flash.vcard_status'));
        }

        // Otherwise, flip status
        $vcard->update([
            'status' => ! $vcard->status,
        ]);

        return $this->sendSuccess(__('messages.flash.vcard_status'));
    }


    public function update(UpdateVcardRequest $request, Vcard $vcard): RedirectResponse
    {
        $url = url($vcard->url_alias);

        $input = $request->all();

        $input['is_locked'] = $request->has('is_locked') ? 1 : 0; // handle unchecked case

        // 🔹 Update vcard details
        $vcard = $this->vcardRepository->update($input, $vcard);
        $user = Auth::user();

        // 🔹 Calculate expiry
        $expiresAt = $this->calculateExpiry($request->expiry_type, $request->custom_days);

        $shareJwt = null;

        $payload = [
            'user_id'  => $user->id,
            'vcard_id' => $vcard->id,
            'iat'      => time(),
            'exp'      => $expiresAt ? $expiresAt->timestamp : null,
        ];

        $shareJwt = JWT::encode($payload, env('JWT_SECRET'), 'HS256');

        // 🔹 Build privacy data safely
        $privacyData = [
            'user_id'    => $user->id,
            'expiry_type' => $request->expiry_type,
            'custom_days' => $request->expiry_type === 'custom' ? $request->custom_days : null,
            'expires_at'  => $expiresAt,
            'share_link'  => $url  . '?share=' .  $shareJwt,
            'share_jwt'   => $shareJwt,
        ];

        // 🔹 Update or create privacy settings for the logged-in user
        UserPrivacySetting::create(
            $privacyData
        );

        // 🔹 Flash success message
        if ($vcard) {
            Session::flash('success', __('messages.flash.vcard_update'));
        }

        // 🔹 Build share link if JWT exists
        $shareLinkUrl = $shareJwt ? $url . '?share=' . $shareJwt : null;

        return redirect()->back()->with('share_link', $shareLinkUrl);
    }


    private function calculateExpiry($type, $custom_days)
    {
        $now = \Carbon\Carbon::now();

        switch ($type) {
            // case '1_hour':
            //     return $now->addHour();
            case '1_hour':
                return $now->addHour();
            case '24_hours':
                return $now->addDay();
            case '7_days':
                return $now->addDays(7);
            case 'custom':
                return $now->addDays($custom_days ?? 1);
            default:
                return null;
        }
    }

    public function destroy(Vcard $vcard): JsonResponse
    {
        $termCondition = TermCondition::whereVcardId($vcard->id)->first();

        if (! empty($termCondition)) {
            $termCondition->delete();
        }

        $privacyPolicy = PrivacyPolicy::whereVcardId($vcard->id)->first();

        if (! empty($privacyPolicy)) {
            $privacyPolicy->delete();
        }

        $vcard->clearMediaCollection(Vcard::PROFILE_PATH);
        $vcard->clearMediaCollection(Vcard::COVER_PATH);
        $vcard->delete();

        $data['make_vcard'] = $this->vcardRepository->checkTotalVcard();

        return $this->sendResponse($data, __('messages.flash.vcard_delete'));
    }

    public function getSlot(Request $request): JsonResponse
    {
        $day = $request->get('day');
        $slots = getSchedulesTimingSlot();
        $html = view('vcards.appointment.slot', ['slots' => $slots, 'day' => $day])->render();

        return $this->sendResponse($html, 'Retrieved successfully.');
    }

    public function getSession(Request $request): JsonResponse
    {
        setLocalLang(getLocalLanguage());

        $vcardId = $request->get('vcardId');

        $date = Carbon::createFromFormat('Y-m-d', $request->date);
        $WeekDaySessions = Appointment::where('day_of_week', ($date->dayOfWeek == 0) ? 7 : $date->dayOfWeek)->where('vcard_id', $vcardId)->get();

        if ($WeekDaySessions->count() == 0) {
            return $this->sendError(__('messages.placeholder.there_is_not_available_slot'));
        }

        $bookedAppointments = ScheduleAppointment::where('vcard_id', $vcardId)->get();

        $bookingSlot = [];
        $bookedSlot = [];
        $userId = Vcard::with('user')->find($vcardId)->user->id;
        foreach ($bookedAppointments as $appointment) {
            if ($appointment->date == $request->date) {
                if (getUserSettingValue('time_format', $userId) == UserSetting::HOUR_24) {
                    $bookedSlot[] = date('H:i', strtotime($appointment->from_time)).' - '.date(
                        'H:i',
                        strtotime($appointment->to_time)
                    );
                } else {
                    $bookedSlot[] = date('h:i A', strtotime($appointment->from_time)).' - '
                        .date('h:i A', strtotime($appointment->to_time));
                }
            }
        }

        foreach ($WeekDaySessions as $index => $WeekDaySession) {
            if (getUserSettingValue('time_format', $userId) == UserSetting::HOUR_24) {
                $bookingSlot[] = date('H:i', strtotime($WeekDaySession->start_time)).' - '.date(
                    'H:i',
                    strtotime($WeekDaySession->end_time)
                );
            } else {
                $bookingSlot[] = date('h:i A', strtotime($WeekDaySession->start_time)).' - '.date(
                    'h:i A',
                    strtotime($WeekDaySession->end_time)
                );
            }
        }

        $slots = array_diff($bookingSlot, $bookedSlot);

        if ($slots == null) {
            return $this->sendError(__('messages.placeholder.there_is_not_available_slot'));
        }

        return $this->sendResponse($slots, 'Retrieved successfully.');
    }

    public function language($languageName, $alias)
    {
        session(['languageChange_'.$alias => $languageName]);
        setLocalLang(getLocalLanguage());

        return $this->sendSuccess(__('messages.flash.language_update'));
    }

    /**
     * @return Application|Factory|View
     */
    public function analytics(Vcard $vcard, Request $request): \Illuminate\View\View
    {
        $input = $request->all();
        $data = $this->vcardRepository->analyticsData($input, $vcard);
        $partName = ($request->part === null) ? 'overview' : $request->part;

        return view('vcards.analytic', compact('vcard', 'partName', 'data'));
    }

    public function chartData(Request $request): JsonResponse
    {
        try {
            $input = $request->all();
            $data = $this->vcardRepository->chartData($input);

            return $this->sendResponse($data, 'Users fetch successfully.');
        } catch (Exception $e) {
            throw new UnprocessableEntityHttpException($e->getMessage());
        }
    }

    /**
     * @return mixed
     */
    public function dashboardChartData(Request $request)
    {
        try {
            $input = $request->all();
            $data = $this->vcardRepository->dashboardChartData($input);

            return $this->sendResponse($data, 'Data fetch successfully.');
        } catch (Exception $e) {
            throw new UnprocessableEntityHttpException($e->getMessage());
        }
    }

    /**
     * @return Application|Factory|View
     */
    public function showBlog($alias, $id): \Illuminate\View\View
    {
        setLocalLang(getLocalLanguage());
        $blog = VcardBlog::with('vcard:id,template_id')->whereRelation('vcard', 'url_alias', '=', $alias)
            ->whereRelation('vcard', 'status', '=', 1)
            ->where('id', $id)
            ->firstOrFail();

        return view('vcards.blog', compact('blog'));
    }

    /**
     * @return Application|Factory|View
     */
    public function showPrivacyPolicy($alias, $id)
    {
        $vacrdTemplate = vcard::find($id);
        setLocalLang(getLocalLanguage());
        $privacyPolicy = PrivacyPolicy::with('vcard')->where('vcard_id', $id)->first();
        $termCondition = TermCondition::with('vcard')->where('vcard_id', $id)->first();
        if ($vacrdTemplate->template_id == 11) {
            return redirect()->route('vcard.show.privacy-policy', [$alias, $id]);
            // return view('vcardTemplates.vcard11.portfolio', compact('privacyPolicy', 'alias', 'termCondition'));
        }

        return view('vcards.privacy-policy', compact('privacyPolicy', 'alias', 'termCondition'));
    }

    public function duplicateVcard($id): JsonResponse
    {
        try {
            $vcard = Vcard::with([
                'services', 'testimonials', 'products', 'blogs', 'privacy_policy', 'term_condition', 'socialLink',
            ])->where('id', $id)->first();
            $this->vcardRepository->getDuplicateVcard($vcard);

            return $this->sendSuccess('Duplicate Vcard Create successfully.');
        } catch (Exception $e) {
            throw new UnprocessableEntityHttpException($e->getMessage());
        }
    }

    public function getUniqueUrlAlias()
    {
        return getUniqueVcardUrlAlias();
    }

    public function checkUniqueUrlAlias($urlAlias)
    {
        $isUniqueUrl = isUniqueVcardUrlAlias($urlAlias);
        if ($isUniqueUrl === true) {
            return $this->sendResponse(['isUnique' => true], 'URL Alias is available to use.');
        }

        $response = ['isUnique' => false, 'usedInVcard' => $isUniqueUrl];

        return $this->sendResponse($response, 'This URL Alias is already in use');
    }

    public function addContact(Vcard $vcard)
    {

        $vcfVcard = new VCardVCard();
        $lastname = $vcard->last_name;
        $firstname = $vcard->first_name;
        $vcfVcard->addName($lastname, $firstname);
        $vcfVcard->addCompany($vcard->company);
        $vcfVcard->addJobtitle($vcard->job_title);

        if (! empty($vcard->email)) {
            // Use library's default method which creates EMAIL;INTERNET format
            // Most contact apps will display this as "Email"
            $vcfVcard->addEmail($vcard->email);
        }

        if (! empty($vcard->alternative_email)) {
            $vcfVcard->addEmail($vcard->alternative_email);
        }

        if (! empty($vcard->phone)) {
            $vcfVcard->addPhoneNumber('+'.$vcard->region_code.$vcard->phone, 'TEL;type=CELL');
        }
        if (! empty($vcard->alternative_phone)) {
            $vcfVcard->addPhoneNumber('+'.$vcard->alternative_region_code.$vcard->alternative_phone, 'TEL;type=Alternate Phone');
        }

        $vcfVcard->addAddress($vcard->location);
        if (! empty($vcard->location_url)) {
            $vcfVcard->addURL($vcard->location_url, 'TYPE=Location URL');
        }

        if(!empty($vcard->whatsapp_number)) {
            $whatsappNumber = !empty($vcard->whatsapp_region_code)
                ? '+' . $vcard->whatsapp_region_code . $vcard->whatsapp_number
                : $vcard->whatsapp_number;


            $vcfVcard->addURL('https://wa.me/' . preg_replace('/[^0-9]/', '', $whatsappNumber), 'TYPE=WhatsApp');
        }
        if(!empty($vcard->telegram_number)) {
            $vcfVcard->addURL('https://t.me/' . ltrim($vcard->telegram_number, '@'), 'TYPE=Telegram');
        }
        $socialLinks = SocialLink::whereVcardId($vcard->id)->first()->toArray();
        $customSocialLinks = SocialIcon::with('media')->whereSocialLinkId($socialLinks['id'])->get();
        unset($socialLinks['id']);
        unset($socialLinks['media']);
        unset($socialLinks['created_at']);
        unset($socialLinks['updated_at']);
        unset($socialLinks['social_icon']);
        unset($socialLinks['vcard_id']);

        foreach ($customSocialLinks as $link) {
            $socialLinks = array_merge($socialLinks, [$link->media[0]['name'] => $link->link]);
        }

        foreach ($socialLinks as $key => $link) {
            $name = Str::camel($key);
            $vcfVcard->addURL($link, 'TYPE='.$name);
        }

        $vcfVcard->addURL(URL::to($vcard->url_alias));

        if ($media = $vcard->getMedia(\App\Models\Vcard::PROFILE_PATH)->first()) {
            $vcfVcard->addPhotoContent(file_get_contents($media->getFullUrl()));
        }

        return \Response::make(
            $vcfVcard->getOutput(),
            200,
            $vcfVcard->getHeaders(true)
        );
    }

    public function showProducts($id,$alias){

        $vcard = Vcard::with([
            'businessHours' => function ($query) {
                $query->where('end_time', '!=', '00:00');
            }, 'services', 'testimonials', 'products', 'blogs', 'privacy_policy', 'term_condition', 'user',
        ])->whereUrlAlias($alias)->first();

        $vcardProducts = $vcard->products->sortDesc()->take(6);

        $products =  Product::with('vcard')->whereVcardId($id)->get();
        $template_id = $products->first()->vcard->template_id;

        if ($vcard->status) {
            return view(
                'vcardTemplates/products/vcard'.$template_id,
                compact(
                    'vcard',
                    'vcardProducts',
                    'products',
                )
            );
        }
    }

    public function deleteImage(Request $request){

        $vcard = Vcard::findOrFail($request->id);

        $type = $request->type;
        $path = null;

        $media = $vcard->getMedia(\App\Models\Vcard::PROFILE_PATH)->first();

        if ($type === 'profile' && $vcard->profile_url) {
            /*$prfileImageName = basename($vcard->profile_url);
            $path = public_path('uploads/vcards/profiles/'. $media->id . '/' . $prfileImageName);
            //$vcard->profile_url = null;*/

            $fullUrl = $vcard->profile_url;
            $relativePath = parse_url($fullUrl, PHP_URL_PATH);
            $absolutePath = public_path($relativePath);

            if (file_exists($absolutePath)) {
                unlink($absolutePath);
            }
            $vcard->clearMediaCollection(Vcard::PROFILE_PATH);


        }

        if ($type === 'cover' && $vcard->cover_url) {
            /*$coverImageName = basename($vcard->cover_url);
            $path = public_path('uploads/vcards/covers/'. $media->id . '/' . $prfileImageName);
            $vcard->cover_url = null;*/

            $fullUrl = $vcard->cover_url;
            $relativePath = parse_url($fullUrl, PHP_URL_PATH);
            $absolutePath = public_path($relativePath);

            if (file_exists($absolutePath)) {
                unlink($absolutePath);
            }

            $vcard->clearMediaCollection(Vcard::COVER_PATH);
        }

        $vcard->save();

        if ($path && file_exists($path)) {
            unlink($path);
        }

        return response()->json(['success' => true, 'message' => 'Image removed successfully']);
    }
     public function exportTemplate()
    {
        return Excel::download(new VcardTemplateExport, 'vcard_template.xlsx');
    }
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,csv,xls',
        ]);

        // Get the current user's subscription
        $subscription = Subscription::where('tenant_id', Auth::user()->tenant_id)
            ->where('status', Subscription::ACTIVE)
            ->first();

        if (!$subscription) {
            return back()->withErrors([
                'import' => 'You do not have an active subscription. Please buy a subscription first.'
            ]);
        }

        $allowedCount = $subscription->no_of_vcards;

        // Count how many vCards the user already has
        $existingCount = Vcard::where('tenant_id', Auth::user()->tenant_id)->count();

        // Read the uploaded file
        $rows = Excel::toArray([], $request->file('file'));

        if (empty($rows) || empty($rows[0])) {
            return back()->withErrors([
                'import' => 'The uploaded file is empty or in an invalid format.'
            ]);
        }

        // Count rows excluding the header
        $importCount = count($rows[0]) - 1;

        // Validate against the allowed number of vCards
        if ($existingCount + $importCount > $allowedCount) {
            return back()->withErrors([
                'import' => 'Sorry, you cannot add more Vcards. Please upgrade your subscription.'
            ]);
        }

        // If validation passes, import
        try {
            Excel::import(new VcardsImport(), $request->file('file'));
            return back()->with('success', 'Data imported successfully.');

        } catch (ValidationException $e) {
            return back()->withErrors(['import' => $e->getMessage()]);
        } catch (Exception $e) {
            return back()->withErrors(['import' => 'Import failed: ' . $e->getMessage()]);
        }
        return back()->with('success', 'Data imported successfully.');
    }

}
