<?php

namespace App\Imports;

use App\Models\Vcard;
use App\Models\SocialLink;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\Importable;
use Maatwebsite\Excel\Concerns\SkipsOnError;
use Maatwebsite\Excel\Concerns\SkipsErrors;
use Illuminate\Validation\ValidationException;

class VcardsImport implements ToModel, WithHeadingRow, SkipsOnError
{
    use Importable, SkipsErrors;

    public $skippedRows = []; // collect skipped rows

    public function model(array $row)
    {
        if ($this->isRowEmpty($row)) {
            return null;
        }

        $alias = $row['url_alias'] ?? null;
        $email = $row['email'] ?? null;

        // ✅ Validate required fields before saving
        if (empty($alias)) {
            throw ValidationException::withMessages([
                'url_alias' => ["Each row must have a 'url_alias'. Import stopped on row with missing alias."]
            ]);
        }
        // ✅ Validate URL alias minimum length (5 characters)
        if (strlen($alias) < 5) {
            throw ValidationException::withMessages([
                'url_alias' => ["URL Alias '{$alias}' must be at least 5 characters long. Import stopped. Please note: URL alias cannot be edited after import."]
            ]);
        }
        
        // ✅ Prevent duplicate alias
        if ($alias && Vcard::where('url_alias', $alias)->exists()) {
            $this->skippedRows[] = [
                'alias' => $alias,
                'reason' => 'Duplicate alias'
            ];
            return null;
        }

        // ✅ Prevent duplicate email
        if ($email && Vcard::where('email', $email)->exists()) {
            $this->skippedRows[] = [
                'email' => $email,
                'reason' => 'Duplicate email'
            ];
            return null;
        }

        $vcard = new Vcard([
            'user_id'     => Auth::id(),
            'url_alias'   => $alias,
            'name'        => $row['vcard_name'] ?? 'null',
            'occupation'  => $row['occupation'] ?? null,
            'description' => $row['description_optional'] ?? null,
            'first_name'  => $row['first_name'] ?? null,
            'last_name'   => $row['last_name'] ?? null,
            'email'       => $email,
            'company'     => $row['company_name'] ?? null,
            'location'    => $row['company_location'] ?? null,
            'location_url'=> $row['location_url_google_map'] ?? null,
        ]);
        $vcard->save();

        SocialLink::create([
            'vcard_id' => $vcard->id,
            'website'  => $row['website_url_optional']   ?? null,
            'linkedin' => $row['linkedin_url_optional']  ?? null,
            'facebook' => $row['facebook_url_optional']  ?? null,
            'instagram'=> $row['instagram_url_optional'] ?? null,
        ]);

        return $vcard;
    }

    private function isRowEmpty(array $row): bool
    {
        $fields = ['first_name','last_name','email','vcard_name'];
        foreach ($fields as $field) {
            if (!empty($row[$field])) {
                return false;
            }
        }
        return true;
    }
}

